/*
* voltage_os_impl.c- Sigmastar
*
* Copyright (c) [2019~2020] SigmaStar Technology.
*
*
* This software is licensed under the terms of the GNU General Public
* License version 2, as published by the Free Software Foundation, and
* may be copied, distributed, and modified under those terms.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License version 2 for more details.
*
*/
#if defined(__KERNEL__)
#include <linux/kernel.h>
#include <linux/of.h>
#include <linux/device.h>
#include <linux/module.h>
#endif
#include "mdrv_gpio_io.h"
#include "registers.h"
#include "ms_platform.h"
#include "voltage_ctrl.h"
#include "voltage_ctrl_demander.h"
#include "voltage_internal.h"
#include "cam_os_wrapper.h"
#include "cam_inter_os.h"

struct device dev_core;

static struct bus_type voltage_subsys = {
    .name = "voltage",
    .dev_name = "voltage",
};

int GetDeviceTreeVidWidth(u32 *width)
{
    struct device_node *np = of_find_node_by_name(NULL, "core_voltage");

    if(np)
    {
        return of_property_read_u32(np, "vid_width", width);
    }
    else
    {
        VOLCTRL_ERR("[Core Voltage] %s: can't get core_voltage node\n", __FUNCTION__);
        return -1;
    }
}

int GetDeviceTreeVidGpiosArray(u32* pins, u32 width)
{
    struct device_node *np = of_find_node_by_name(NULL, "core_voltage");

    if(np)
    {
        return of_property_read_variable_u32_array(np, "vid_gpios", pins, 0, width);
    }
    else
    {
        VOLCTRL_ERR("[Core Voltage] %s: can't get core_voltage node\n", __FUNCTION__);
        return -1;
    }
}

int GetDeviceTreeVidVoltagesArray(u32* voltages, u32 width)
{
    struct device_node *np = of_find_node_by_name(NULL, "core_voltage");

    if(np)
    {
        return of_property_read_variable_u32_array(np, "vid_voltages", voltages, 0, (1 << width));
    }
    else
    {
        VOLCTRL_ERR("[Core Voltage] %s: can't get core_voltage node\n", __FUNCTION__);
        return -1;
    }
}

static ssize_t show_scaling_voltage(struct device *dev, struct device_attribute *attr, char *buf)
{
    char *str = buf;
    char *end = buf + PAGE_SIZE;

    str += scnprintf(str, end - str, "%d\n", EnableScalingVoltageGet());

    return (str - buf);
}

static ssize_t store_scaling_voltage(struct device *dev,  struct device_attribute *attr, const char *buf, size_t count)
{
    u32 enable;
    if (sscanf(buf, "%d", &enable)<=0)
        return 0;

    if(enable)
    {
        EnableScalingVoltageSet(1);
        VOLCTRL_DBG("[Core Voltage] %s: scaling ON\n", __FUNCTION__);
    }
    else
    {
        EnableScalingVoltageSet(0);
        VOLCTRL_DBG("[Core Voltage] %s: scaling OFF\n", __FUNCTION__);
    }

    sync_core_voltage();

    return count;
}

static ssize_t show_voltage_available(struct device *dev, struct device_attribute *attr, char *buf)
{
    char *str = buf;
    char *end = buf + PAGE_SIZE;
    unsigned int gpio_bitmap = 0;
    unsigned int *pins = NULL;
    unsigned int pin_num = 0;
    unsigned int *voltages = NULL;
    unsigned int voltage_num = 0;
    u8 i,j;

    str += scnprintf(str, end - str, "\tVcore(mV)");

    if(core_voltage_pin(&pins, &pin_num) == 0)
    {
        for (i = 0; i < pin_num; i++)
        {
            str += scnprintf(str, end - str, "\tvid_%d(%d)", i, pins[i]);
        }
    }
    else
    {
        goto out;
    }

    str += scnprintf(str, end - str, "\n");

    if (pin_num)
    {
        if(core_voltage_available(&voltages, &voltage_num) == 0)
        {
            for (j = 0; j < voltage_num; j++, gpio_bitmap++)
            {
                str += scnprintf(str, end - str, "[%d]\t%u", j, voltages[j]);

                for(i=0;i<pin_num;i++)
                {
                    str += scnprintf(str, end - str, "\t\t%u", (gpio_bitmap & 1 << i)? 1 : 0);
                }
                str += scnprintf(str, end - str, "\n");
            }
        }
    }

out:
    return (str - buf);
}

static ssize_t show_voltage_current(struct device *dev, struct device_attribute *attr, char *buf)
{
    char *str = buf;
    char *end = buf + PAGE_SIZE;
    unsigned int i;
    unsigned int *pins = NULL;
    unsigned int pin_num = 0;
#ifdef CONFIG_SS_DUALOS
    unsigned int query_voltage=0;
#endif

    core_voltage_pin(&pins, &pin_num);

    if (pin_num)
    {
#ifdef CONFIG_SS_DUALOS
        query_voltage = CamInterOsSignal(INTEROS_SC_L2R_CORE_VOLTAGE_GET, 0, 0, 0);
        str += scnprintf(str, end - str, "%u\n", query_voltage);
#else
        str += scnprintf(str, end - str, "%d\n", get_core_voltage());
#endif
    }
    else
        str += scnprintf(str, end - str, "Unknown (not assigned GPIO)\n");

    for (i=0; i<VOLTAGE_DEMANDER_MAX; i++)
    {
#ifdef CONFIG_SS_DUALOS
        query_voltage = CamInterOsSignal(INTEROS_SC_L2R_CORE_VOLTAGE_GET, 1, i, 0);
        if (query_voltage)
        {
            str += scnprintf(str, end - str, "    %-32s%d\n", DemanderNameGet(i), query_voltage);
        }
#else
        if (DemanderRequestValueGet(i))
            str += scnprintf(str, end - str, "    %-32s%d\n", DemanderNameGet(i), DemanderRequestValueGet(i));
#endif
        else
            str += scnprintf(str, end - str, "    %-32s-\n", DemanderNameGet(i));
    }

    return (str - buf);
}

static ssize_t store_voltage_current(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    u32 voltage = 0;
    if (sscanf(buf, "%d", &voltage)<=0)
        return 0;

    if (EnableScalingVoltageGet())
        set_core_voltage(VOLTAGE_DEMANDER_USER, voltage);
    else
        CamOsPrintf("[Core Voltage] voltage scaling not enable\n");

    return count;
}

static ssize_t show_vid_gpio_map(struct device *dev, struct device_attribute *attr, char *buf)
{
    char *str = buf;
    char *end = buf + PAGE_SIZE;
    unsigned int i = 0;
    unsigned int *pins = NULL;
    unsigned int pin_num = 0;

    if(core_voltage_pin(&pins, &pin_num) == 0)
    {
        for (i = 0; i < pin_num; i++)
        {
            str += scnprintf(str, end - str, "vid_%d=%d ", i, pins[i]);
        }
    }

    str += scnprintf(str, end - str, "\n");

    return (str - buf);
}

#if defined(__KERNEL__)
#ifdef CONFIG_SS_VOLTAGE_CTRL_WITH_OSC
EXPORT_SYMBOL(sync_core_voltage_with_OSC_and_TEMP);
#endif
EXPORT_SYMBOL(set_core_voltage);
EXPORT_SYMBOL(get_core_voltage);
EXPORT_SYMBOL(core_voltage_available);
EXPORT_SYMBOL(core_voltage_pin);

DEVICE_ATTR(scaling_voltage, 0644, show_scaling_voltage, store_scaling_voltage);
DEVICE_ATTR(voltage_available, 0444, show_voltage_available, NULL);
DEVICE_ATTR(voltage_current, 0644, show_voltage_current, store_voltage_current);
DEVICE_ATTR(vid_gpio_map, 0444, show_vid_gpio_map, NULL);
#endif

int voltage_control_interface_init(void)
{
    int ret = 0;

    dev_core.kobj.name = "core";
    dev_core.bus = &voltage_subsys;

    ret = subsys_system_register(&voltage_subsys, NULL);
    if (ret) {
        CamOsPrintf(KERN_ERR "Failed to register voltage sub system!! %d\n",ret);
        goto voltage_control_interface_init_err;
    }

    ret = device_register(&dev_core);
    if (ret) {
        CamOsPrintf(KERN_ERR "Failed to register voltage core device!! %d\n",ret);
        goto voltage_control_interface_init_err;
    }

    device_create_file(&dev_core, &dev_attr_scaling_voltage);
    device_create_file(&dev_core, &dev_attr_voltage_available);
    device_create_file(&dev_core, &dev_attr_voltage_current);
    device_create_file(&dev_core, &dev_attr_vid_gpio_map);

    return 0;

voltage_control_interface_init_err:
    return -1;
}
